-- Safe index creation script that checks for existing indexes first
-- This approach works with all MySQL versions

-- For MySQL 8.0+, you can use:
-- CREATE INDEX IF NOT EXISTS idx_name ON table_name(column_name);

-- For older MySQL versions, use this approach:

-- First, check existing indexes:
-- SHOW INDEX FROM posts;
-- SHOW INDEX FROM categories;
-- SHOW INDEX FROM post_tags;
-- SHOW INDEX FROM attachments;

-- Then manually create only the missing indexes:

-- Index on posts.category_id for faster joins
-- CREATE INDEX idx_posts_category_id ON posts(category_id);

-- Index on posts.slug for faster lookups
-- CREATE INDEX idx_posts_slug ON posts(slug);

-- Index on posts.created_at for faster sorting
-- CREATE INDEX idx_posts_created_at ON posts(created_at);

-- Index on categories.slug for faster lookups
-- CREATE INDEX idx_categories_slug ON categories(slug);

-- Index on post_tags.post_id for faster joins
-- CREATE INDEX idx_post_tags_post_id ON post_tags(post_id);

-- Index on post_tags.tag_id for faster joins
-- CREATE INDEX idx_post_tags_tag_id ON post_tags(tag_id);

-- Index on attachments.post_id for faster joins
-- CREATE INDEX idx_attachments_post_id ON attachments(post_id);

-- Index on attachments.sort_order for faster sorting
-- CREATE INDEX idx_attachments_sort_order ON attachments(sort_order);

-- Index on attachments.uploaded_at for faster sorting
-- CREATE INDEX idx_attachments_uploaded_at ON attachments(uploaded_at);

-- Composite index for category counting query
-- CREATE INDEX idx_posts_category_count ON posts(category_id, id);

-- Alternative approach: Use this procedure to safely create indexes (MySQL 5.7+)

DELIMITER $$

CREATE PROCEDURE AddIndexIfNotExists(
    IN schema_name VARCHAR(128),
    IN table_name VARCHAR(128),
    IN index_name VARCHAR(128),
    IN column_name VARCHAR(128)
)
BEGIN
    DECLARE index_exists INT DEFAULT 0;
    
    SELECT COUNT(*) INTO index_exists
    FROM information_schema.statistics
    WHERE table_schema = schema_name
    AND table_name = table_name
    AND index_name = index_name;
    
    IF index_exists = 0 THEN
        SET @sql = CONCAT('CREATE INDEX ', index_name, ' ON ', table_name, ' (', column_name, ')');
        PREPARE stmt FROM @sql;
        EXECUTE stmt;
        DEALLOCATE PREPARE stmt;
    END IF;
END$$

DELIMITER ;

-- Usage example (uncomment to use):
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'posts', 'idx_posts_category_id', 'category_id');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'posts', 'idx_posts_slug', 'slug');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'posts', 'idx_posts_created_at', 'created_at');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'categories', 'idx_categories_slug', 'slug');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'post_tags', 'idx_post_tags_post_id', 'post_id');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'post_tags', 'idx_post_tags_tag_id', 'tag_id');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'attachments', 'idx_attachments_post_id', 'post_id');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'attachments', 'idx_attachments_sort_order', 'sort_order');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'attachments', 'idx_attachments_uploaded_at', 'uploaded_at');
-- CALL AddIndexIfNotExists('archive_dudu2_2', 'posts', 'idx_posts_category_count', 'category_id, id');

-- Drop the procedure after use:
-- DROP PROCEDURE AddIndexIfNotExists;